---
title: Managed networks · Cloudflare Zero Trust docs
description: Cloudflare WARP allows you to selectively apply specific device
  profiles and WARP client settings when a device connects to a secure network
  location, such as an office. WARP identifies these managed networks by
  detecting a TLS endpoint you set up on the network.
lastUpdated: 2025-10-13T21:03:08.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/managed-networks/
  md: https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/managed-networks/index.md
---

Feature availability

| [WARP modes](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/warp-modes/) | [Zero Trust plans](https://www.cloudflare.com/teams-pricing/) |
| - | - |
| All modes | All plans |

| System | Availability | Minimum WARP version |
| - | - | - |
| Windows | ✅ | 2025.1.861.0 |
| macOS | ✅ | 2025.1.861.0 |
| Linux | ✅ | 2025.1.861.0 |
| iOS | ✅ | 1.0 |
| Android | ✅ | 1.0 |
| ChromeOS | ✅ | 1.0 |

Cloudflare WARP allows you to selectively apply specific [device profiles](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/device-profiles/) and WARP client settings when a device connects to a secure network location, such as an office. WARP identifies these managed networks by detecting a TLS endpoint you set up on the network.

On this page, you will learn how to:

* Create a TLS endpoint on your trusted network.
* Configure the TLS endpoint in Zero Trust to set up a managed network.
* Apply the appropriate device profile to a device when the WARP client detects it is on your managed network.

## Requirements

* The WARP client scans for managed networks when the operating system's default route changes, the SSID of the active Wi-Fi connection changes, or the DNS servers of the default interface change. To minimize performance impact, reuse the same TLS endpoint across multiple locations unless you require distinct settings profiles for each location.
* Ensure that the device can only reach one managed network at any given time. If multiple managed networks are configured and reachable, there is no way to determine which settings profile the device will receive.

## WARP client managed network detection

When you configure a managed network, the WARP client uses the TLS endpoint to determine whether the device is on that network.

The time it takes to apply the correct device profile depends on how quickly the TLS endpoint responds.

If the TLS endpoint times out after 5 seconds, the WARP client will determine that the device is not on a managed network and will apply the default device profile. The WARP client only retries detection if a non-timeout error occurs. A timeout triggers fallback to the default device profile without further retries.

## 1. Choose a TLS endpoint

A TLS endpoint is a host on your network that serves a TLS certificate. The TLS endpoint acts like a network location beacon — when a device connects to a network, the WARP client on the device detects the TLS endpoint and validates the TLS certificate against the SHA-256 fingerprint (if specified) or against the local certificate store to check that it is signed by a public certificate authority.

The TLS certificate can be hosted by any device on your network. However, the endpoint must be inaccessible to users outside of the network location. WARP will automatically exclude the managed network endpoint from all device profiles to ensure that users cannot connect to this endpoint over Cloudflare Tunnel. We recommend choosing a host that is physically in the office which remote users do not need to access, such as a printer.

### Create a new TLS endpoint

If you do not already have a TLS endpoint on your network, you can set one up as follows:

1. Generate a TLS certificate:

   ```sh
   openssl req -x509 -newkey rsa:4096 -sha256 -days 3650 -nodes -keyout example.key -out example.pem -subj "/CN=example.com" -addext "subjectAltName=DNS:example.com"
   ```

   The command will output a certificate in PEM format and its private key. Store these files in a secure place.

   Note

   The WARP client requires certificates to include `CN` and `subjectAltName` metadata. You can use `example.com` or any other domain.

2. Configure an HTTPS server on your network to use this certificate and key. The example below demonstrates how to serve the TLS certificate from an nginx container in Docker:

   a. Create an nginx configuration file called `nginx.conf`:

   ```txt
   events {
   worker_connections  1024;
   }


   http {
       server {
         listen              443 ssl;
         ssl_certificate     /certs/example.pem;
         ssl_certificate_key /certs/example.key;
         location / {
               return 200;
         }
       }
   }
   ```

   If needed, replace `/certs/example.pem` and `/certs/example.key` with the locations of your certificate and key.

   b. Add the nginx image to your Docker compose file:

   ```yml
   version: "3.3"
   services:
   nginx:
     image: nginx:latest
     ports:
       - 3333:443
     volumes:
       - ./nginx.conf:/etc/nginx/nginx.conf:ro
       - ./certs:/certs:ro
   ```

   If needed, replace `./nginx.conf` and `./certs` with the locations of your nginx configuration file and certificate.

   c. Start the server:

   ```sh
   docker-compose up -d
   ```

3. To test that the TLS server is working, run a curl command from the end user's device:

   ```sh
   curl --verbose --insecure https://<private-server-IP>:3333/
   ```

   You need to pass the `--insecure` option because we are using a self-signed certificate. If the device is connected to the network, the request should return a `200` status code.

Windows IIS

To create a TLS endpoint using Windows Internet Information Services (IIS) Manager:

1. Run Powershell as administrator.

2. Generate a self-signed certificate:

   ```powershell
   New-SelfSignedCertificate -CertStoreLocation Cert:\LocalMachine\My -DnsName "office-name.example.internal" -FriendlyName "Cloudflare Managed Network Certificate" -NotAfter (Get-Date).AddYears(10)
   ```

   ```powershell
     PSParentPath: Microsoft.PowerShell.Security\Certificate::LocalMachine\My


   Thumbprint                                Subject
   ----------                                -------
   0660C4FCD15F69C49BD080FEEA4136B3D302B41B  CN=office-name.example.internal
   ```

3. Extract the certificate's SHA-256 fingerprint:

   ```powershell
   [System.BitConverter]::ToString([System.Security.Cryptography.SHA256]::Create().ComputeHash((Get-ChildItem Cert:\LocalMachine\My | Where-Object { $_.FriendlyName -eq "Cloudflare Managed Network Certificate" }).RawData)) -replace "-", ""
   ```

   ```powershell
   DD4F4806C57A5BBAF1AA5B080F0541DA75DB468D0A1FE731310149500CCD8662
   ```

   You will need the SHA-256 fingerprint to [configure the managed network in Zero Trust](#3-add-managed-network-to-zero-trust). Do not use the default SHA-1 thumbprint generated by the `New-SelfSignedCertificate` command.

4. Open IIS Manager.

5. In the **Connections** pane, right-click the **Sites** node and select **Add Website**.

6. In **Site name**, enter any name for the TLS server (for example, `Managed Network Server`).

7. In **Physical path**, enter any directory that contains a `.htm` or `html` file, such as `C:\inetpub\wwwroot`. Cloudflare does not validate the content within the directory.

8. Under **Binding**, configure the following fields:

   * **Type**: *https*
   * **IP address**: *All Unassigned*
   * **Port**: `443`
   * **Host name**: Enter the certificate's Common Name (CN). The CN of our example certificate is `office-name.example.internal`.
   * **Require Server Name Indication**: Enabled
   * **SSL certificate**: Select the name of your TLS certificate. Our example certificate is called `Cloudflare Managed Network Certificate`.

9. To test that the TLS server is working, run a curl command from the end user's device:

   ```sh
   curl --verbose --insecure --resolve office-name.example.internal:443:<private-server-IP> https://office-name.example.internal
   ```

   You need to pass the `--insecure` option because we are using a self-signed certificate. The `--resolve` option allows you to connect to the server's private IP but also pass the hostname to the server for SNI and certificate validation. If the device is connected to the network, the request should return your directory's default homepage (`C:\inetpub\wwwroot\iisstart.htm`).

### Supported cipher suites

The WARP client establishes a TLS connection using [Rustls](https://github.com/rustls/rustls). Make sure your TLS endpoint accepts one of the [cipher suites supported by Rustls](https://docs.rs/rustls/0.21.10/src/rustls/suites.rs.html#125-143).

## 2. Extract the SHA-256 fingerprint

The SHA-256 fingerprint is only required if your TLS endpoint uses a self-signed certificate.

* Local certificate

  To obtain the SHA-256 fingerprint of a local certificate:

  ```sh
  openssl x509 -noout -fingerprint -sha256 -inform pem -in example.pem | tr -d :
  ```

  The output will look something like:

  ```txt
  SHA256 Fingerprint=DD4F4806C57A5BBAF1AA5B080F0541DA75DB468D0A1FE731310149500CCD8662
  ```

* Remote server

  To test connectivity and obtain the SHA-256 fingerprint of a remote server:

  ```sh
  openssl s_client -connect <private-server-IP>:443 < /dev/null 2> /dev/null | openssl x509 -noout -fingerprint -sha256 | tr -d :
  ```

  The output will look something like:

  ```txt
  SHA256 Fingerprint=DD4F4806C57A5BBAF1AA5B080F0541DA75DB468D0A1FE731310149500CCD8662
  ```

## 3. Add managed network to Zero Trust

* Dashboard

  1. In [Zero Trust](https://one.dash.cloudflare.com), go to **Settings** > **WARP Client**.

  2. Scroll down to **Network locations** and select **Add new**.

  3. Name your network location.

  4. In **Host and Port**, enter the private IP address and port number of your [TLS endpoint](#create-a-new-tls-endpoint) (for example, `192.168.185.198:3333`).

     Note

     We recommend using the private IP of your managed network endpoint and not a hostname to prevent issues related to DNS lookups resolving the incorrect IP.

  5. (Optional) In **TLS Cert SHA-256**, enter the [SHA-256 fingerprint](#2-extract-the-sha-256-fingerprint) of the TLS certificate. This field is only needed for self-signed certificates. If a TLS fingerprint is not supplied, WARP validates the certificate against the local certificate store and checks that it is signed by a public certificate authority.

* Terraform (v5)

  1. Add the following permission to your [`cloudflare_api_token`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/api_token):

     * `Zero Trust Write`

  2. Add a managed network using the [`cloudflare_zero_trust_device_managed_network`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/zero_trust_device_managed_network) resource:

     ```tf
     resource "cloudflare_zero_trust_device_managed_networks" "office" {
       account_id = var.cloudflare_account_id
       name       = "Office managed network"
       type       = "tls"
       config = {
         tls_sockaddr = "192.168.185.198:3333"
         sha256       = "DD4F4806C57A5BBAF1AA5B080F0541DA75DB468D0A1FE731310149500CCD8662"
       }
     }
     ```

WARP will automatically exclude the TLS endpoint from all device profiles if it is specified as a private IP address. This exclusion prevents remote users from accessing the endpoint through the WARP tunnel on any port. If the TLS endpoint is specified as a hostname instead of a private IP, WARP will not automatically exclude it.

If a device profile uses [Split Tunnels](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/route-traffic/split-tunnels/) in **Include** mode, ensure that the Split Tunnel entries do not contain the TLS endpoint IP address; otherwise, the entire IP range will be excluded from the WARP tunnel.

## 4. Configure device profile

* Dashboard

  1. In [Zero Trust](https://one.dash.cloudflare.com), go to **Settings** > **WARP Client**.

  2. Under **Profile settings**, create a new [settings profile](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/device-profiles/) or edit an existing profile.

  3. To apply this profile whenever a device connects to your network, add the following rule:

     | Selector | Operator | Value |
     | - | - | - |
     | Managed network | is | `<NETWORK-NAME>` |

  4. Save the profile.

* Terraform (v5)

  In [`cloudflare_zero_trust_device_custom_profile`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/zero_trust_device_custom_profile), configure a `match` expression using the `network` selector. For example, the following device profile will match all devices connected a specific managed network:

  ```tf
  resource "cloudflare_zero_trust_device_custom_profile" "office" {
    account_id            = var.cloudflare_account_id
    name                  = "Office"
    description           = "Devices connected to the office network"
    precedence            = 1
    service_mode_v2       = {mode = "warp"}


    match = trimspace(replace(<<-EOT
      network == "${cloudflare_zero_trust_device_managed_networks.office.name}"
    EOT
    , "\n", " "))
  }
  ```

Managed networks are now enabled. Every time a device in your organization connects to a network (for example, when waking up the device or changing Wi-Fi networks), the WARP client will determine its network location and apply the corresponding settings profile.

## 5. Verify managed network

To check if the WARP client detects the network location:

1. Turn on WARP.
2. Disconnect and reconnect to the network.
3. Open a terminal and run `warp-cli debug alternate-network`.

## Related resources

* [Device profiles](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/device-profiles/) - How to create and manage the device profiles you apply via managed networks.
* [WARP settings](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/configure-warp/warp-settings/) - Defines how WARP behaves and what users can do.
* [WARP troubleshooting guide](https://developers.cloudflare.com/cloudflare-one/connections/connect-devices/warp/troubleshooting/troubleshooting-guide/) - Troubleshoot common WARP issues.
